
/*

  Packet with many different algorithms, which may
  be combined

  */

#include "conf.h"
#include "fft.h"
#include "profile.h"

#ifndef M_PI
#define M_PI PI
#endif

#define FFT_PRINT_OFF

/* Prototype for internal use */
void _fft_mixradix(FFT_TYPE *real,FFT_TYPE *imag,int base);

void fft_radix2_step(FFT_TYPE *preal,FFT_TYPE *pimag,const int size,const int step) {
  register int i,j,k,alpha,u,alpha2,alpha05;
  register FFT_TYPE *wkreal,*wkimag,*real,*imag;
  register FFT_TYPE ar,ai,br,bi,cr,ci;

  alpha=(2<<step);
  alpha2=2*alpha;
  alpha05=alpha/2;
  u=2*FFT_MAXSIZE/alpha;
  wkreal=fft_COS; /* cos */
  wkimag=fft_SIN;
  for (i=0;i<2*size;i+=alpha2) {
    real=preal+i;
    imag=pimag+i;
    /* calculation of c=b*Wk obsolete, Wk=1.0+0.0*j */
    /* ar,ai=psymb */
    ar=*real;
    ai=*imag;
    br=real[alpha];
    bi=imag[alpha];
    *real=(ar+br)/2;
    *imag=(ai+bi)/2;
    real[alpha]=(ar-br)/2;
    imag[alpha]=(ai-bi)/2;
    k=u;
    for (j=1;j<alpha05;j++) {
      real+=2;
      imag+=2;
      /* br,bi=psymb+alpha */
      br=real[alpha];
      bi=imag[alpha];
      /* ar,ai=Wk */
      ar=wkreal[k];
      ai=wkimag[k];

      /* calculating c=b*Wk */
      cr=MUL_SHIFT( ((ar*br)-(ai*bi)) );
      ci=MUL_SHIFT( ((ar*bi)+(ai*br)) );

      /* ar,ai=psymb */
      ar=*real; /* +1 rounding? */
      ai=*imag; /* +1 rounding? */
      *real=(ar+cr)/2;
      *imag=(ai+ci)/2;
      real[alpha]=(ar-cr)/2;
      imag[alpha]=(ai-ci)/2;
      k+=u;
    }
  }
}

/*  a mix between radix-2 and -4. while operands are sorted bitreversaled for
    radix-2, this routine performs a radix-4 operation-step. Compared with
    radix-4, only the operands 1 & 2 of 0..3 are switched (alpha & alpha2).
    this MUST be considered in the twiddle-step */
void fft_mixradix_faststep(FFT_TYPE *preal,FFT_TYPE *pimag,const int size,const int step) {
  register int i,j,alpha1,alpha2,alpha3,alpha4,alpha05;
  register FFT_TYPE a,t1,t2,t3,t4,t5,t6,t7,t8; /* temporary */
  register FFT_TYPE *real,*imag;

  alpha1=2*(1<<(2*step)); /* 2*exp(log(4)*step) */
  alpha2=2*alpha1;
  alpha3=3*alpha1;
  alpha4=4*alpha1;
  alpha05=alpha1/2;
  for (i=0;i<2*size;i+=alpha4) {
    real=preal+i;
    imag=pimag+i;
    for (j=alpha05;j!=0;j--) {
      /* operand fetch and calculation */
      t1=(*real); /* ar1 +2 rounding? */
      t3=(*imag); /* ai1 +2 rounding? */
      a=real[alpha1];   /* ar3 */
      t2=t1-a;      /* ar1-ar3 */
      t1+=a;        /* ar1+ar3 */
      a=imag[alpha1]; /* ai3 */
      t4=t3-a;      /* ai1-ai3 */
      t3+=a;        /* ai1+ai3 */
      t5=real[alpha2];    /* ar2 */
      t7=imag[alpha2];  /* ai2 */
      a=real[alpha3];    /* ar4 */
      t6=t5-a;       /* ar2-ar4 */
      t5+=a;         /* ar2+ar4 */
      a=imag[alpha3];  /* ai4 */
      t8=t7-a;       /* ai2-ai4 */
      t7+=a;         /* ai2+ai4 */

      *real=(t1+t5);          /* store b1 */
      *imag=(t3+t7);
      real[alpha3]=(t2-t8);          /* store b4 */
      imag[alpha3]=(t4+t6);
      real[alpha1]=(t2+t8);          /* store b2 */
      imag[alpha1]=(t4-t6);
      real[alpha2]=(t1-t5);          /* store b3 */
      imag[alpha2]=(t3-t7);

      real+=2;
      imag+=2;
    }
  }
}

/* radix-4 twiddle-algorithm for radix-2 fft 
   only for the first twiddling step! all symetries used */
void fft_mixradix_twiddle0(FFT_TYPE *preal,FFT_TYPE *pimag,const int size) {
  register int i;
  register FFT_TYPE tmp,ar,ai;
  register FFT_TYPE *real,*imag;
  register FFT_TYPE wkreal1_16th,wkreal3_16th,wkreal9_16th;
  register FFT_TYPE wkimag1_16th,wkimag3_16th,wkimag9_16th;

  wkreal1_16th=fft_COS[FFT_MAXSIZE/16];
  wkimag1_16th=fft_SIN[FFT_MAXSIZE/16];
  wkreal3_16th=fft_COS[3*FFT_MAXSIZE/16];
  wkimag3_16th=fft_SIN[3*FFT_MAXSIZE/16];
  wkreal9_16th=fft_COS[9*FFT_MAXSIZE/16];
  wkimag9_16th=fft_SIN[9*FFT_MAXSIZE/16];

  /* k=0,1,2,3 j=0,2,1,3 [j1,j2]=00,10,01,11 bitreversaled
     k*j ungleich Null: 10,   12,   14,   18,   20,   22,   26,   28,   30
                   j*k: 1*2   2*2   3*2   1*1   2*1   3*1   1*3   2*3   3*3 
                   phi: N/8   N/4  3N/8   N/16  N/8  3N/16 3N/16 3N/8  9N/16
     Wk= exp( -2*PI*k*j/N *sqrt(-1)) = Wk[phi] + Wk[phi+1]*sqrt(-1)   */
  for(i=0;i<2*size;i+=32) {
    real=preal+i;
    imag=pimag+i;

    ar=real[10];ai=imag[10]; /* 1/8 N */
    real[10]=MUL_SHIFT( ((ar+ai)*onedivsqrt2) );
    imag[10]=MUL_SHIFT( ((ai-ar)*onedivsqrt2) );

    tmp=-real[12]; /* 1/2 N */
    real[12]=imag[12];
    imag[12]=tmp;

    ar=real[14];ai=imag[14]; /* 3/8 N */
    real[14]=MUL_SHIFT( ((ai-ar)*onedivsqrt2) );
    imag[14]=MUL_SHIFT( ((-ar-ai)*onedivsqrt2) );

    ar=real[18];ai=imag[18];
    real[18]=MUL_SHIFT( ((ar*wkreal1_16th)-(ai*wkimag1_16th)) );
    imag[18]=MUL_SHIFT( ((ar*wkimag1_16th)+(ai*wkreal1_16th)) );

    ar=real[20];ai=imag[20]; /* 1/8 N */
    real[20]=MUL_SHIFT( ((ar+ai)*onedivsqrt2) );
    imag[20]=MUL_SHIFT( ((ai-ar)*onedivsqrt2) );

    ar=real[22];ai=imag[22];
    real[22]=MUL_SHIFT( ((ar*wkreal3_16th)-(ai*wkimag3_16th)) );
    imag[22]=MUL_SHIFT( ((ar*wkimag3_16th)+(ai*wkreal3_16th)) );

    ar=real[26];ai=imag[26];
    real[26]=MUL_SHIFT( ((ar*wkreal3_16th)-(ai*wkimag3_16th)) );
    imag[26]=MUL_SHIFT( ((ar*wkimag3_16th)+(ai*wkreal3_16th)) );

    ar=real[28];ai=imag[28];
    real[28]=MUL_SHIFT( ((ai-ar)*onedivsqrt2) );
    imag[28]=MUL_SHIFT( ((-ar-ai)*onedivsqrt2) );

    ar=real[30];ai=imag[30];
    real[30]=MUL_SHIFT( ((ar*wkreal9_16th)-(ai*wkimag9_16th)) );
    imag[30]=MUL_SHIFT( ((ar*wkimag9_16th)+(ai*wkreal9_16th)) );
  }
}

/* twiddle routine, k*j=0,N/8,N/4 symetries extracted */
void fft_mixradix_twiddleN(FFT_TYPE *preal,FFT_TYPE *pimag,const int size,const int step) {
  register int i=0;
  int j,k,v,u;
  int nn,n2,n05;
  int radix,radix2,radix3;
  FFT_TYPE *wkreal,*wkimag,*real,*imag;
  FFT_TYPE ar,ai,br,bi;

  wkreal=fft_COS; /* cos */
  wkimag=fft_SIN; /* sin */
  nn=4*(1<<(step*2));
  radix=FFT_MAXSIZE/(2*nn);
  radix2=2*radix;
  radix3=3*radix;
  v=5*FFT_MAXSIZE/4;
  n2=nn*2;
  n05=nn/2;
  real=preal;
  imag=pimag;
#ifdef FFT_PRINT
  printf("twiddle step%d nn%d v%d radix%d\n",step,nn,v,radix);
#endif

  for(i=0;i<2*size;i+=(8*nn)) {
    /* k=0/j=0 is obsolete */

    /* k=1 j<N/2 */
    k=i+n2+2;
    u=radix2;
    /* k=2 j<N/4 */
    for(j=1;j<n05;j++) {
      ar=real[k];ai=imag[k];
      br=wkreal[u];bi=wkimag[u];
      real[k]=MUL_SHIFT( ((ar*br)-(ai*bi)) );
      imag[k]=MUL_SHIFT( ((ar*bi)+(ai*br)) );
      k+=2;
      u+=radix2;
    }
    /* k=2 j==N/2 */
    ar=-real[k];
    real[k]=imag[k];
    imag[k]=ar;
    k+=2;
    u+=radix2;
    /* k=2 N/2<j<N*3/4 */
    for(j=1;j<n05;j++) {
      ar=real[k];ai=imag[k];
      br=wkreal[u];bi=wkimag[u];
      real[k]=MUL_SHIFT( ((ar*br)-(ai*bi)) );
      imag[k]=MUL_SHIFT( ((ar*bi)+(ai*br)) );
      k+=2;
      u+=radix2;
    }
    u=radix;
    k+=2;
    for(j=1;j<nn;j++) {
      ar=real[k];ai=imag[k];
      br=wkreal[u];bi=wkimag[u];
      real[k]=MUL_SHIFT( ((ar*br)-(ai*bi)) );
      imag[k]=MUL_SHIFT( ((ar*bi)+(ai*br)) );
      k+=2;
      u+=radix;
    }
    u=radix3;
    k+=2;
    /* k=3 j<N/2 */
    for(j=1;j<nn;j++) {
      ar=real[k];ai=imag[k];
      br=wkreal[u];bi=wkimag[u];
      real[k]=MUL_SHIFT( ((ar*br)-(ai*bi)) );
      imag[k]=MUL_SHIFT( ((ar*bi)+(ai*br)) );
      k+=2;
      u+=radix3;
      u%=v;
    }
  }
}

/* all purpose twiddle routine, only k=0|j=0:=>Wk=1 symetry used.
   this routine serves for demonstration purposes. cause it is slower
   than twiddle0/twiddleN it is never used
*/
void fft_mixradix_twiddle(FFT_TYPE *preal,FFT_TYPE *pimag,const int size,const int step) {
  register int tmp,i,j,k,nn,n2,v,u;
  register int radix,radix2,radix3;
  register FFT_TYPE *real,*imag,*wkreal,*wkimag;
  register FFT_TYPE ar,ai,br,bi;

  wkreal=fft_COS; /* cos */
  wkimag=fft_SIN; /* sin */
  nn=4*(1<<(step*2));
  n2=2*nn;
  radix=FFT_MAXSIZE/(2*nn);
  radix2=2*radix;
  radix3=3*radix;
  v=2*FFT_MAXSIZE;
  real=preal;
  imag=pimag;
#ifdef FFT_PRINT
  printf("twiddle step%d nn%d v%d radix%d\n",step,nn,v,radix);
#endif
  for(i=0;i<2*size;i+=(8*nn)) {
    k=i+n2+2;
    u=radix2;
    /* J*K,K=2 */
    for(j=1;j<nn;j++) {
      ar=real[k];ai=imag[k];
      br=wkreal[u];bi=wkimag[u];
      real[k]=MUL_SHIFT( ((ar*br)-(ai*bi)) );
      imag[k]=MUL_SHIFT( ((ar*bi)+(ai*br)) );
      k+=2;
      u+=radix2;
    }
    /* K=1 */
    u=radix;
    k+=2;
    for(j=1;j<nn;j++) {
      ar=real[k];ai=imag[k];
      br=wkreal[u];bi=wkimag[u];
      real[k]=MUL_SHIFT( ((ar*br)-(ai*bi)) );
      imag[k]=MUL_SHIFT( ((ar*bi)+(ai*br)) );
      k+=2;
      u+=radix;
    }
    u=radix3;
    k+=2;
    /* K=3 */
    for(j=1;j<nn;j++) {
      ar=real[k];ai=imag[k];
      br=wkreal[u];bi=wkimag[u];
      real[k]=MUL_SHIFT( ((ar*br)-(ai*bi)) );
      imag[k]=MUL_SHIFT( ((ar*bi)+(ai*br)) );
      k+=2;
      u+=radix3;
      u%=v;
    }
  }
}

/* base is log(size)/log(2) */
void fft_mixradix(FFT_TYPE *symb,int base) {
#ifdef FFT_PRINT
  printf("fft\n");
#endif
#ifdef COFDM_PROFILE
  prof_start(profile_fft);
#endif
  _fft_mixradix(symb,symb+1,base);
#ifdef COFDM_PROFILE
  prof_stop();
#endif
}

void ifft_mixradix(FFT_TYPE *symb,int base) {
#ifdef FFT_PRINT
  printf("ifft\n");
#endif
#ifdef COFDM_PROFILE
  prof_start(profile_ifft);
#endif
  _fft_mixradix(symb+1,symb,base);
#ifdef COFDM_PROFILE
  prof_stop();
#endif
}

void _fft_mixradix(FFT_TYPE *real,FFT_TYPE *imag,int base) {
  int size,i,j,quadsteps,dualstep;
  
  quadsteps=base/2;
  dualstep=base%2;

  size=(1<<base);
  fft_mixradix_faststep(real,imag,size,0);
#ifdef FFT_PRINT
  printf("quadstep 0\n");
#endif  
  fft_mixradix_twiddle0(real,imag,size);
#ifdef FFT_PRINT
  printf("twiddlestep 0\n");
#endif  
  for (i=1;i<quadsteps-1;i++) {
    fft_mixradix_faststep(real,imag,size,i);
#ifdef FFT_PRINT
    printf("quadstep %d\n",i);
#endif  
    fft_mixradix_twiddleN(real,imag,size,i);
#ifdef FFT_PRINT
    printf("twiddlestep %d\n",i);
#endif  
  }
  fft_mixradix_faststep(real,imag,size,i);
  if (dualstep>0) fft_radix2_step(real,imag,size,base-1);
}

int mul_int30(int a,int b) {
  register int msb,lsb1,lsb2;

  msb=(a>>15)*(b>>15);
  lsb1=(a>>15)*(b&0x00007fff);
  lsb2=(a&0x00007fff)*(b>>15);

  return(msb+((lsb1+lsb2+(1<<14))/(1<<15)));
}

